# coding=utf-8
# Copyright 2023-present the HuggingFace Inc. team.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

import warnings
from typing import List, Optional

import bitsandbytes as bnb
import torch

from peft.import_utils import is_bnb_4bit_available, is_bnb_available
from peft.utils.other import transpose

from .layer import LoCALayer, idct_2d_impl


if is_bnb_available():

    class Linear8bitLt(torch.nn.Module, LoCALayer):
        # loca implemented in a dense layer
        def __init__(
            self,
            base_layer: torch.nn.Module,
            adapter_name: str,
            n_frequency: int = 0,
            scale: float = 0.1,
            loca_dropout: float = 0.0,
            loca_dct_mode: str = 'default',
            fan_in_fan_out: bool = False,
            init_loca_weights: bool = True,
            learn_location_iter: int =1000,
            **kwargs,
        ) -> None:
            super().__init__()
            LoCALayer.__init__(self, base_layer,**kwargs)
            self.fan_in_fan_out = fan_in_fan_out
            self.update_layer(adapter_name, n_frequency, scale, loca_dropout, learn_location_iter, loca_dct_mode, init_loca_weights)

        def merge(self, safe_merge: bool = False, adapter_names: Optional[List[str]] = None) -> None:
            """
            Merge the active adapter weights into the base weights

            Args:
                safe_merge (`bool`, *optional*):
                    If True, the merge operation will be performed in a copy of the original weights and check for NaNs
                    before merging the weights. This is useful if you want to check if the merge operation will produce
                    NaNs. Defaults to `False`.
                adapter_names (`List[str]`, *optional*):
                    The list of adapter names that should be merged. If None, all active adapters will be merged.
                    Defaults to `None`.
            """
            if self.merged:
                warnings.warn(
                    f"Already following adapters were merged {','.join(self.merged_adapters)}. "
                    f"You are now additionally merging {','.join(self.active_adapters)}."
                )

            if adapter_names is None:
                adapter_names = self.active_adapters

            for active_adapter in adapter_names:
                if active_adapter in self.spectrum.keys():
                    warnings.warn(
                        "Merge loca module to 8-bit linear may get different generations due to rounding errors."
                    )
                    base_layer = self.get_base_layer()
                    weight = base_layer.weight
                    state = base_layer.state

                    if safe_merge:
                        # Note that safe_merge will be slower than the normal merge
                        # because of the copy operation.
                        orig_weights = weight.data.clone()
                        loca_data = self.get_delta_weight(active_adapter)

                        # Dequantize the result of identity matrix and int8 weight
                        im = torch.eye(orig_weights.shape[-1]).contiguous().half().to(orig_weights.device)
                        im, imt, SCim, SCimt, coo_tensorim = bnb.functional.double_quant(im)
                        im, Sim = bnb.functional.transform(im, "col32")
                        if state.CxB is None:
                            state.CxB, state.SB = bnb.functional.transform(orig_weights.data, to_order=state.formatB)
                        out32, Sout32 = bnb.functional.igemmlt(im, state.CxB, Sim, state.SB)
                        output = bnb.functional.mm_dequant(out32, Sout32, SCim, state.SCB, bias=None).t()

                        w_data = output.to(loca_data.dtype).to(loca_data.device) + loca_data
                        
                        if not torch.isfinite(w_data).all():
                            raise ValueError(
                                f"NaNs detected in the merged weights. The adapter {active_adapter} seems to be broken"
                            )

                        base_layer.weight = bnb.nn.Int8Params(
                            w_data.to("cpu"), requires_grad=False, has_fp16_weights=weight.has_fp16_weights
                        ).to(weight.device)
                    else:
                        loca_data = self.get_delta_weight(active_adapter)

                        # Dequantize the result of identity matrix and int8 weight
                        im = torch.eye(weight.shape[-1]).contiguous().half().to(weight.device)
                        im, imt, SCim, SCimt, coo_tensorim = bnb.functional.double_quant(im)
                        im, Sim = bnb.functional.transform(im, "col32")
                        if state.CxB is None:
                            state.CxB, state.SB = bnb.functional.transform(weight.data, to_order=state.formatB)
                        out32, Sout32 = bnb.functional.igemmlt(im, state.CxB, Sim, state.SB)
                        output = bnb.functional.mm_dequant(out32, Sout32, SCim, state.SCB, bias=None).t()

                        w_data = output.to(loca_data.dtype).to(loca_data.device) + loca_data
                        if not torch.isfinite(w_data).all():
                            raise ValueError(
                                f"NaNs detected in the merged weights. The adapter {active_adapter} seems to be broken"
                            )

                        base_layer.weight = bnb.nn.Int8Params(
                            w_data.to("cpu"), requires_grad=False, has_fp16_weights=weight.has_fp16_weights
                        ).to(weight.device)

                    state.reset_grads()
                    self.merged_adapters.append(active_adapter)

        def unmerge(self) -> None:
            """
            This method unmerges all merged adapter layers from the base weights.
            """
            if not self.merged:
                warnings.warn("Already unmerged. Nothing to do.")
                return

            while len(self.merged_adapters) > 0:
                active_adapter = self.merged_adapters.pop()
                if active_adapter not in self.spectrum.keys():
                    continue
                warnings.warn(
                    "Unmerge loca module from 8-bit linear may get different generations due to rounding errors."
                )
                
                base_layer = self.get_base_layer()
                weight = base_layer.weight
                state = base_layer.state

                loca_data = self.get_delta_weight(active_adapter)

                # Dequantize the result of identity matrix and int8 weight
                im = torch.eye(weight.shape[-1]).contiguous().half().to(weight.device)
                im, imt, SCim, SCimt, coo_tensorim = bnb.functional.double_quant(im)
                im, Sim = bnb.functional.transform(im, "col32")

                if state.CxB is None:
                    state.CxB, state.SB = bnb.functional.transform(weight.data, to_order=state.formatB)
                out32, Sout32 = bnb.functional.igemmlt(im, state.CxB, Sim, state.SB)
                output = bnb.functional.mm_dequant(out32, Sout32, SCim, state.SCB, bias=None).t()

                w_data = output.to(loca_data.dtype).to(loca_data.device) - loca_data

                base_layer.weight = bnb.nn.Int8Params(
                    w_data.to("cpu"), requires_grad=False, has_fp16_weights=weight.has_fp16_weights
                ).to(weight.device)

                state.reset_grads()

        def spectrum_to_para(self, adapter):
            spectral_para = self.spectrum[adapter]
            lo_clip = torch.clamp(self.spectrum_indices[adapter], min=0, max=1)
            lo_clip[0,:], lo_clip[1,:] = lo_clip[0,:] * (self.out_features -1), lo_clip[1,:] * (self.in_features -1)
            if self.spectrum_indices[adapter].requires_grad:
                return self.sparse_idct2d_with_index_gradient(spectral_para, lo_clip, self.out_features, self.in_features)
            else:
                return idct_2d_impl(spectral_para, lo_clip.round().long(), self.out_features, self.in_features, self.loca_dct_mode)
            
        def get_delta_weight(self, adapter):  
            def T(w):
                return w.T if self.fan_in_fan_out else w      
            return T(self.spectrum_to_para(adapter)) * self.scale[adapter]

        def forward(self, x: torch.Tensor, *args, **kwargs) -> torch.Tensor:
            if self.training:
                if self.global_iter == 0:
                    # print("start learning coefficients:",self.global_iter)
                    self.spectrum['default'].requires_grad = True
                    self.spectrum_indices['default'].requires_grad = False
                else:
                    if self.global_iter < self.learn_location_iter:
                        cycle_position = self.global_iter % 30    
                        if cycle_position == 0:
                            # print("start learning coefficients:", self.global_iter)
                            self.spectrum['default'].requires_grad = True
                            self.spectrum_indices['default'].requires_grad = False
                        if cycle_position == 10:
                            # print("start learning positions:", self.global_iter)
                            self.spectrum['default'].requires_grad = False
                            self.spectrum_indices['default'].requires_grad = True
                    if self.global_iter == self.learn_location_iter:
                        # leran coefficients thereafter
                        self.spectrum['default'].requires_grad = True
                        self.spectrum_indices['default'].requires_grad = False                
                        # print("spectrum_indices:", self.spectrum_indices['default'])
                self.global_iter += 1
            previous_dtype = x.dtype

            if self.disable_adapters:
                if self.merged:
                    self.unmerge()
                result = self.base_layer(x, *args, **kwargs)
            elif self.merged:
                result = self.base_layer(x, *args, **kwargs)
            else:
                result = self.base_layer(x, *args, **kwargs)
                for active_adapter in self.active_adapters:
                    if active_adapter not in self.spectrum.keys():
                        continue
                    # print(self.spectrum[active_adapter].grad)
                    dropout = self.loca_dropout_layer[active_adapter]
                    delta_w = self.get_delta_weight(active_adapter)
                    result += dropout(x) @ delta_w.T

            result = result.to(previous_dtype)
            return result

        def __repr__(self) -> str:
            rep = super().__repr__()
            return "loca." + rep


if is_bnb_4bit_available():

    class Linear4bit(torch.nn.Module, LoCALayer):
        # Loca implemented in a dense layer
        def __init__(
            self,
            base_layer: torch.nn.Module,
            adapter_name: str,
            n_frequency: int = 0,
            scale: float = 0.1,
            loca_dropout: float = 0.0,
            loca_dct_mode: str = 'default',
            fan_in_fan_out: bool = False,
            init_loca_weights: bool = True,
            learn_location_iter: int =1000,
            **kwargs,
        ) -> None:
            super().__init__()
            LoCALayer.__init__(self, base_layer,**kwargs)
            self.fan_in_fan_out = fan_in_fan_out
            self.update_layer(adapter_name, n_frequency, scale, loca_dropout, learn_location_iter, loca_dct_mode, init_loca_weights)

        def merge(self, safe_merge: bool = False, adapter_names: Optional[List[str]] = None) -> None:
            """
            Merge the active adapter weights into the base weights

            Args:
                safe_merge (`bool`, *optional*):
                    If True, the merge operation will be performed in a copy of the original weights and check for NaNs
                    before merging the weights. This is useful if you want to check if the merge operation will produce
                    NaNs. Defaults to `False`.
                adapter_names (`List[str]`, *optional*):
                    The list of adapter names that should be merged. If None, all active adapters will be merged. Defaults
                    to `None`.
            """
            if self.merged:
                warnings.warn(
                    f"Already following adapters were merged {','.join(self.merged_adapters)}. "
                    f"You are now additionally merging {','.join(self.active_adapters)}."
                )

            if adapter_names is None:
                adapter_names = self.active_adapters

            for active_adapter in adapter_names:
                if active_adapter in self.spectrum.keys():
                    base_layer = self.get_base_layer()
                    if safe_merge:
                        # Note that safe_merge will be slower than the normal merge
                        # because of the copy operation.
                        orig_weights = base_layer.weight.data.clone()
                        kwargs = orig_weights.__dict__
                        loca_data = self.get_delta_weight(active_adapter)
                        w_data = bnb.functional.dequantize_4bit(orig_weights.data, orig_weights.quant_state) + loca_data
                    
                        if not torch.isfinite(w_data).all():
                            raise ValueError(
                                f"NaNs detected in the merged weights. The adapter {active_adapter} seems to be broken"
                            )

                        base_layer.weight.data = bnb.nn.Params4bit(w_data.to("cpu"), requires_grad=False, **kwargs).to(
                        orig_weights.device
                    )
                    else:
                        weight = self.get_base_layer().weight
                        kwargs = weight.__dict__
                        loca_data = self.get_delta_weight(active_adapter)

                        w_data = bnb.functional.dequantize_4bit(weight.data, weight.quant_state) + loca_data
                        if not torch.isfinite(w_data).all():
                            raise ValueError(
                                f"NaNs detected in the merged weights. The adapter {active_adapter} seems to be broken"
                            )

                        self.get_base_layer().weight = bnb.nn.Params4bit(w_data.to("cpu"), requires_grad=False, **kwargs).to(
                            weight.device
                        )
                    self.merged_adapters.append(active_adapter)

        def unmerge(self) -> None:
            """
            This method unmerges all merged adapter layers from the base weights.
            """
            if not self.merged:
                warnings.warn("Already unmerged. Nothing to do.")
                return

            while len(self.merged_adapters) > 0:
                active_adapter = self.merged_adapters.pop()
                if active_adapter not in self.spectrum.keys():
                    continue
                warnings.warn(
                    "Unmerge loca module to 4-bit linear may get different generations due to rounding errors."
                )
                weight = self.get_base_layer().weight
                kwargs = weight.__dict__
                loca_data = self.get_delta_weight(active_adapter)
                w_data = bnb.functional.dequantize_4bit(weight.data, weight.quant_state) - loca_data
                self.get_base_layer().weight = bnb.nn.Params4bit(w_data.to("cpu"), requires_grad=False, **kwargs).to(
                    weight.device
                )

        def spectrum_to_para(self, adapter):
            spectral_para = self.spectrum[adapter]
            lo_clip = torch.clamp(self.spectrum_indices[adapter], min=0, max=1)
            lo_clip[0,:], lo_clip[1,:] = lo_clip[0,:] * (self.out_features -1), lo_clip[1,:] * (self.in_features -1)
            if self.spectrum_indices[adapter].requires_grad:
                return self.sparse_idct2d_with_index_gradient(spectral_para, lo_clip, self.out_features, self.in_features)
            else:
                return idct_2d_impl(spectral_para, lo_clip.round().long(), self.out_features, self.in_features, self.loca_dct_mode)
            
        def get_delta_weight(self, adapter):  
            def T(w):
                return w.T if self.fan_in_fan_out else w      
            return T(self.spectrum_to_para(adapter)) * self.scale[adapter]

        def forward(self, x: torch.Tensor, *args, **kwargs) -> torch.Tensor:
            if self.training:
                if self.global_iter == 0:
                    # print("start learning coefficients:",self.global_iter)
                    self.spectrum['default'].requires_grad = True
                    self.spectrum_indices['default'].requires_grad = False
                else:
                    if self.global_iter < self.learn_location_iter:
                        cycle_position = self.global_iter % 30    
                        if cycle_position == 0:
                            # print("start learning coefficients:", self.global_iter)
                            self.spectrum['default'].requires_grad = True
                            self.spectrum_indices['default'].requires_grad = False
                        if cycle_position == 10:
                            # print("start learning positions:", self.global_iter)
                            self.spectrum['default'].requires_grad = False
                            self.spectrum_indices['default'].requires_grad = True
                    if self.global_iter == self.learn_location_iter:
                        # leran coefficients thereafter
                        self.spectrum['default'].requires_grad = True
                        self.spectrum_indices['default'].requires_grad = False                
                        # print("spectrum_indices:", self.spectrum_indices['default'])
                self.global_iter += 1
            previous_dtype = x.dtype

            if self.disable_adapters:
                if self.merged:
                    self.unmerge()
                result = self.base_layer(x, *args, **kwargs)
            elif self.merged:
                result = self.base_layer(x, *args, **kwargs)
            else:
                result = self.base_layer(x, *args, **kwargs)
                for active_adapter in self.active_adapters:
                    if active_adapter not in self.spectrum.keys():
                        continue
                    # print(self.spectrum[active_adapter].grad)
                    dropout = self.loca_dropout_layer[active_adapter]
                    delta_w = self.get_delta_weight(active_adapter)
                    result += dropout(x) @ delta_w.T

            result = result.to(previous_dtype)
            return result

        def __repr__(self) -> str:
            rep = super().__repr__()
            return "loca." + rep
